const jsonpointer = require("jsonpointer");

class Validator {
  constructor(ajv, schema, customFormat) {
    if (ajv && customFormat) {
      Object.keys(customFormat).forEach(function (format) {
        const validate = customFormat[format].validate;
        const fn = function (s) {
          try {
            return validate(s);
          } catch (e) {
            return false;
          }
        };
        ajv.addFormat(format, fn);
      });
    }
    this.ajv = ajv;
    this.schema = schema;
    this.customFormat = customFormat || {};
    this.fn = this.ajv.compile(this.schema);
  }
  execute(data, normalize) {
    const response = { data: data };
    response.valid = this.fn(data);
    response.errors = response.valid ? [] : this.fn.errors;

    for (const e of response.errors.filter((a) => a.keyword === "format")) {
      try {
        const prev = jsonpointer.get(data, e.instancePath);
        this.customFormat[e.params.format].validate(prev);
      } catch (g) {
        e.error = g.message;
      }
    }

    if (normalize) {
      const copy = JSON.parse(JSON.stringify(data));
      response.changed = false;
      response.normalized = copy;
      for (const e of response.errors.filter((a) => a.keyword === "format")) {
        const prev = jsonpointer.get(copy, e.instancePath);
        const next = this.customFormat[e.params.format].normalize(prev);
        if (prev !== next) {
          jsonpointer.set(copy, e.instancePath, next);
          response.changed = true;
          e.prev = prev;
          e.next = next;
        }
      }
    }
    return response;
  }
  validate(data) {
    return this.execute(data, false).valid;
  }
  normalize(data) {
    return this.execute(data, true).normalized;
  }
}

module.exports = Validator;
