const test = require("tape");
const Validator = require("../validator.js");
const Ajv = require("ajv");

const simple = new Validator(new Ajv({ allErrors: true }), {
  type: "object",
  properties: {
    id: { type: "string" },
    name: { type: "string" },
  },
});

const custom = new Validator(
  new Ajv({ allErrors: true }),
  {
    type: "object",
    properties: {
      code: { type: "string", format: "iso3166-1-alpha3" },
    },
  },
  {
    "iso3166-1-alpha3": {
      validate: function (a) {
        if (a.match(/^[A-Z]{3}$/)) return true;
        throw new Error("iso3166-1-alpha3 は半角大文字アルファベット3文字です");
      },
      normalize: function (a) {
        const mapping = { JP: "JPN" };
        const n = a.normalize("NFKC").toUpperCase();
        return mapping[n] || n;
      },
    },
  }
);

test("simple#validate", function (t) {
  t.ok(
    simple.validate({ id: "1", name: "John" }),
    "一般的なスキーマで valid な JSON を valid と判定できる"
  );
  t.end();
});

test("custom#validate", function (t) {
  t.ok(custom.validate({ code: "JPN" }), "有効なコード(alpha-3 半角大文字) を valid と判定できる");
  t.notOk(
    custom.validate({ code: "jpn" }),
    "無効なコード(alpha-3 半角小文字)を invalid と判定できる"
  );
  t.notOk(
    custom.validate({ code: "ｊｐｎ" }),
    "無効なコード(alpha-3 全角小文字) を invalid と判定できる"
  );
  t.notOk(
    custom.validate({ code: "ＪＰＮ" }),
    "無効なコード(alpha-3 全角大文字) を invalid と判定できる"
  );
  t.notOk(
    custom.validate({ code: "JP" }),
    "無効なコード(alpha-2 半角大文字) を invalid と判定できる"
  );
  t.notOk(
    custom.validate({ code: "jp" }),
    "無効なコード(alpha-2 半角小文字) を invalid と判定できる"
  );
  t.notOk(
    custom.validate({ code: "ｊｐ" }),
    "無効なコード(alpha-2 全角小文字) を invalid と判定できる"
  );
  t.notOk(
    custom.validate({ code: "ＪＰ" }),
    "無効なコード(alpha-2 全角大文字) を invalid と判定できる"
  );
  t.end();
});

test("custom#normalize", function (t) {
  t.deepEqual(
    custom.normalize({ code: "JPN" }),
    { code: "JPN" },
    "有効なコード(alpha-3 半角大文字) はそのまま出力される"
  );
  t.deepEqual(
    custom.normalize({ code: "jpn" }),
    { code: "JPN" },
    "無効なコード(alpha-3 半角小文字) を正規化できる"
  );
  t.deepEqual(
    custom.normalize({ code: "ｊｐｎ" }),
    { code: "JPN" },
    "無効なコード(alpha-3 全角小文字) を正規化できる"
  );
  t.deepEqual(
    custom.normalize({ code: "ＪＰＮ" }),
    { code: "JPN" },
    "無効なコード(alpha-3 全角大文字) を正規化できる"
  );
  t.deepEqual(
    custom.normalize({ code: "JP" }),
    { code: "JPN" },
    "無効なコード(alpha-2 半角大文字) を正規化できる"
  );
  t.deepEqual(
    custom.normalize({ code: "jp" }),
    { code: "JPN" },
    "無効なコード(alpha-2 半角小文字) を正規化できる"
  );
  t.deepEqual(
    custom.normalize({ code: "ｊｐ" }),
    { code: "JPN" },
    "無効なコード(alpha-2 全角小文字) を正規化できる"
  );
  t.deepEqual(
    custom.normalize({ code: "ＪＰ" }),
    { code: "JPN" },
    "無効なコード(alpha-2 全角大文字) を正規化できる"
  );
  t.deepEqual(
    custom.normalize({ code: "日本" }),
    { code: "日本" },
    "正規化対象外の無効な値はそのまま出力されること"
  );

  t.end();
});

test("custom#execute", function (t) {
  t.deepEqual(
    custom.execute({ code: "JPN" }, false),
    { data: { code: "JPN" }, valid: true, errors: [] },
    "正常系データの execute(data,normalize=false) の結果は所定の data,valid=true,errors=[] を保持すること"
  );
  t.deepEqual(
    custom.execute({ code: "JPN" }, true),
    { data: { code: "JPN" }, valid: true, errors: [], changed: false, normalized: { code: "JPN" } },
    "正常系データの execute(data,normalize=true) の結果は所定の data,valid=true,errors=[] を保持すること"
  );
  t.deepEqual(
    custom.execute({ code: "jpn" }, false),
    {
      data: { code: "jpn" },
      valid: false,
      errors: [
        {
          instancePath: "/code",
          schemaPath: "#/properties/code/format",
          keyword: "format",
          params: { format: "iso3166-1-alpha3" },
          message: 'must match format "iso3166-1-alpha3"',
          error: "iso3166-1-alpha3 は半角大文字アルファベット3文字です",
        },
      ],
    },
    "異常系データの execute(data,normalize=false) の結果は所定の data,valid=false,errors=[*日本語化エラー] を保持すること"
  );
  t.deepEqual(
    custom.execute({ code: "jpn" }, true),
    {
      data: { code: "jpn" },
      valid: false,
      errors: [
        {
          instancePath: "/code",
          schemaPath: "#/properties/code/format",
          keyword: "format",
          params: { format: "iso3166-1-alpha3" },
          message: 'must match format "iso3166-1-alpha3"',
          error: "iso3166-1-alpha3 は半角大文字アルファベット3文字です",
          prev: "jpn",
          next: "JPN",
        },
      ],
      changed: true,
      normalized: { code: "JPN" },
    },
    "異常系データの execute(data,normalize=true) の結果は所定の data,valid=false,errors=[*日本語化エラー],changed=true,normalized={*} を保持すること"
  );

  t.end();
});
