# GIF コンポーネントツール: データバリデーションコンポーネント

JSON データに対して、JSON schema に基づいた検査・修正を行うライブラリです。

# 開発者向け情報

## Quick Start

ここでは `dev-gif-component-tools-validator-1.0.0.tgz` が `${SOMEWHERE}` フォルダにあるものとします。

1. 適当な作業フォルダを作成
2. 作業フォルダで `dev-gif-component-tools-validator-1.0.0.tgz` を展開
3. `dev-gif-component-tools-validator-1.0.0` フォルダでインストール
4. 作業フォルダに 自作アプリ用の `myapp` フォルダを作成
5. `myapp` フォルダで [ajv](https://www.npmjs.com/package/ajv) と validator をインストール
6. `myapp/main.js` を作成・編集
7. `myapp/main.js` を実行

具体的な実行手順は以下の通りです。

```
$ mkdir work
$ cd work
$ tar xvzf ${SOMEWHERE}/dev-gif-component-tools-validator-1.0.0.tgz
$ pushd dev-gif-component-tools-validator-1.0.0
$ npm install
$ popd
$ mkdir myapp
$ cd myapp
$ npm install ajv ../dev-gif-component-tools-validator-1.0.0
$ vi main.js
$ node main.js
true
false
$
```

`main.js` には以下の内容を記述します。

```main.js
const Validator = require("@dev-gif-component-tools/validator");
const Ajv = require("ajv");
const ajv = new Ajv();
const schema = {
  type: "object",
  required : ["id"],
  properties: {
    id: { type: "string" },
    name: { type: "string" },
  },
};
const v = new Validator(ajv, schema);
console.log(v.validate({ id: "1", name: "John" })); // true
console.log(v.validate({ name: "Alice" })); // false
```

## 使用例

### Step1. Ajv のインスタンスの生成

Ajv のインスタンスを生成します。

```step1-1.js
const ajv = new Ajv();
```

必要に応じてオプションを設定します。

```step1-2.js
const ajv = new Ajv({ allErrors: true });
```

### Step2. Validator の生成

ajv のインスタンスと JSON Schema を与えて `Validator` のインスタンスを作成します。
`v.validate(data)` メソッドを実行すると
スキーマに適合する場合は `true`, そうでない場合は `false` が返ります。

```step2-1.js
const Validator = require("@dev-gif-component-tools/validator");
const Ajv = require("ajv");
const ajv = new Ajv();
const schema = {
  type: "object",
  required : ["id"],
  properties: {
    id: { type: "string" },
    name: { type: "string" },
  },
};
const v = new Validator(ajv, schema);
console.log(v.validate({ id: "1", name: "John" })); // true
console.log(v.validate({ name: "Alice" })); // false

```

### Step3. カスタムフォーマットの追加と正規化

ajv のインスタンスとカスタムフォーマットの定義された JSON Schema を与えて、
`Validator` のインスタンスを作成します。
このとき、第三引数として
`{"custom-format-name" : {validate:function(s){...},normalize:function(s){...}} ...}`
のような関数群を与えます。

`v.normalize(data)` を実行すると
スキーマへの適合が検査とともにカスタムフォーマットのバリデーションと正規化が行われ、
正規化された結果が返ります。

```normalize.js
const Validator = require("@dev-gif-component-tools/validator");
const Ajv = require("ajv");
const ajv = new Ajv({ allErrors: true });
const schema = {
  type: "object",
  properties: {
    code: { type: "string", format: "iso3166-alpha-3"},
  },
};
const v = new Validator(ajv,schema,{
  "iso3166-alpha-3" : {
    validate: function (a) {
      if (a.match(/^[A-Z]{3}$/)) return true;
        throw new Error("iso3166-1-alpha3 は半角大文字アルファベット3文字です");
      },
      normalize: function (a) {
        const mapping = { JP: "JPN" };
        const n = a.normalize("NFKC").toUpperCase();
        return mapping[n] || n;
      },
    }
});
console.log(v.normalize({ code: "JPN" })); // {code: "JPN"}
console.log(v.normalize({ code: "jpn" })); // {code: "JPN"}
console.log(v.normalize({ code: "ｊｐｎ" })); // {code: "JPN"}
console.log(v.normalize({ code: "JP" })); // {code: "JPN"}
console.log(v.normalize({ code: "jp" })); // {code: "JPN"}
console.log(v.normalize({ code: "ｊｐ" })); // {code: "JPN"}

```

### Step4. より詳細な検証結果の取得

`v.validate(data)` は `boolean` を返しますが、
代わりに `v.execute(data,false)` をコールすることで以下のような構造化データが取得できます。

```validate-result.json
{
  "data": { "code": "jpn" },
  "valid": false,
  "errors": [
    {
      "instancePath": "/code",
      "schemaPath": "#/properties/code/format",
      "keyword": "format",
      "params": { "format": "iso3166-1-alpha3" },
      "message": "must match format 'iso3166-1-alpha3'",
      "error": "iso3166-1-alpha3 は半角大文字アルファベット3文字です"
    }
  ]
}
```

- `data` 検証対象の入力データ
- `valid` スキーマに対する適合性
- `errors` Ajv の validate.errors に対して拡張エラーメッセージ `error` を追加したもの

### Step5. より詳細な正規化結果の取得

`v.normalize(data)` は正規化結果のインスタンスを返しますが、
代わりに `v.execute(data,true)` をコールすることで以下のような構造化データが取得できます。

```normalize-result.json
{
  "data": { "code": "jpn" },
  "valid": false,
  "errors": [
    {
      "instancePath": "/code",
      "schemaPath": "#/properties/code/format",
      "keyword": "format",
      "params": { "format": "iso3166-1-alpha3" },
      "message": "must match format 'iso3166-1-alpha3'",
      "error": "iso3166-1-alpha3 は半角大文字アルファベット3文字です",
      "prev": "jpn",
      "next": "JPN"
    }
  ],
  "changed": true,
  "normalized": { "code": "JPN" }
}
```

- `data` 検証対象の入力データ
- `valid` スキーマに対する適合性
- `errors` Ajv の validate.errors に対して拡張エラーメッセージ `error`、正規化前文字列、 `prev` 正規化後文字列 `next` を追加したもの
- `changed` 正規化の有無
- `normalized` 正規化されたインスタンス

## Test

以下の手順でライブラリに対するテストを実施できます。

```
$ mkdir temp
$ cd temp
$ tar xvzf ${SOMEWHERE}/dev-gif-component-tools-validator-1.0.0.tgz
$ cd dev-gif-component-tools-validator-1.0.0
$ npm install
$ npm test

> @dev-gif-component-tools/validator@1.0.0 test
> tape test/test-*.js

TAP version 13
# simple#validate
ok 1 一般的なスキーマで valid な JSON を valid と判定できる
# custom#validate
ok 2 有効なコード(alpha-3 半角大文字) を valid と判定できる
ok 3 無効なコード(alpha-3 半角小文字)を invalid と判定できる
ok 4 無効なコード(alpha-3 全角小文字) を invalid と判定できる
(中略)

1..22
# tests 22
# pass  22

# ok

$
```

また、以下のコマンドでカバレッジを計測できます。

```
$ npm run coverage
(中略)
# ok

--------------|---------|----------|---------|---------|-------------------
File          | % Stmts | % Branch | % Funcs | % Lines | Uncovered Line #s
--------------|---------|----------|---------|---------|-------------------
All files     |     100 |      100 |     100 |     100 |
 validator.js |     100 |      100 |     100 |     100 |
--------------|---------|----------|---------|---------|-------------------
$
```

(以上)
