# util

GIF で使用されるデータ型（リテラル）に対する検証と正規化のための関数群を提供するライブラリです。

# 基本設計

## セマンティックスキーマの mock.js

`dev-gif-component-tools-schema-1.0.0.tgz` では GIF のための JSON Schema (セマンティックスキーマ) が整備されます。
同ファイル内の `mock.js` では以下のように、フォーマット名に対して
`validate` と `normalize` 関数を持つオブジェクトを用意しておくことで
セマンティックスキーマが正しくコンパイルされること、
また `dev-gif-component-tools-validator-1.0.0.tgz` での正規化機能の動作を保証しています。

```mock.js
module.exports = {
 abstract : {
      validate: function (a) {
        return true;
      },
      normalize: function (a) {
        return a.normalize("NFC");
      }
  }
,
 accessMethod : {
      validate: function (a) {
        return true;
      },
      normalize: function (a) {
        return a.normalize("NFC");
      }
  }
,
}
```

上記の `mock.js` は `validate` に対して `true`, `normalize` に対して Unicode 正規化 (NFC) を実行するだけのモック実装ですが、本ライブラリこれらに対して実際の実装を提供します。

## format と実装のバインド

本ライブラリのエントリポイントとなる `gif.js` では
以下のような形で `mock.js` の実装を提供します。

```gif.js
module.exports = {
  abstract: require("./lib/ex/string.js"),
  accessMethod: require("./lib/ex/string.js"),
  accompaniedByGuideDogsAndServiceDogsStatus: require("./lib/pcd/StatusCodeList.js"),
  activityStatus: require("./lib/ex/string.js"),
  ...
};
```

フォーマットと実装の対応は下表の通りです。

| format                                     | implementation                                                           |
| ------------------------------------------ | ------------------------------------------------------------------------ |
| abstract                                   | [lib/ex/string.js](#libexstringjs)                                       |
| accessMethod                               | [lib/ex/string.js](#libexstringjs)                                       |
| accompaniedByGuideDogsAndServiceDogsStatus | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| activityStatus                             | [lib/ex/string.js](#libexstringjs)                                       |
| age                                        | [lib/ex/integer.js](#libexintegerjs)                                     |
| alternateName                              | [lib/ex/string.js](#libexstringjs)                                       |
| alternativeName                            | [lib/ex/string.js](#libexstringjs)                                       |
| alternativeNameEn                          | [lib/ex/hankakuAlphaNumeric.js](#libexhankakualphanumericjs)             |
| alternativeNameKana                        | [lib/ex/zenkakuKana.js](#libexzenkakukanajs)                             |
| applicationDeadlineTime                    | [lib/ex/time.js](#libextimejs)                                           |
| applicationMethod                          | [lib/ex/string.js](#libexstringjs)                                       |
| applicationStartTime                       | [lib/ex/time.js](#libextimejs)                                           |
| applicationUrl                             | [lib/ex/uri.js](#libexurijs)                                             |
| areaCode                                   | [lib/ex/string.js](#libexstringjs)                                       |
| availabileDateTimeDescription              | [lib/ex/string.js](#libexstringjs)                                       |
| availabileDateTimeFormula                  | [lib/ex/dateTimeFormula.js](#libexdatetimeformulajs)                     |
| availableDay                               | [lib/ex/day.js](#libexdayjs)                                             |
| bicycleParkingInformation                  | [lib/ex/string.js](#libexstringjs)                                       |
| brailleAndReadingAssistanceStatus          | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| buildingFootprintArea_MTK                  | [lib/ex/number.js](#libexnumberjs)                                       |
| buildingName                               | [lib/ex/string.js](#libexstringjs)                                       |
| buildingNameEtc                            | [lib/ex/string.js](#libexstringjs)                                       |
| buildingNameEtcEn                          | [lib/ex/stringEn.js](#libexstringenjs)                                   |
| buildingNameEtcKana                        | [lib/ex/stringKana.js](#libexstringkanajs)                               |
| caneRentalStatus                           | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| capacity                                   | [lib/ex/integer.js](#libexintegerjs)                                     |
| capacityRemark                             | [lib/ex/string.js](#libexstringjs)                                       |
| capital_JPY                                | [lib/ex/integer.js](#libexintegerjs)                                     |
| categoryOfBusiness                         | [lib/ex/string.js](#libexstringjs)                                       |
| categoryOfOrganization                     | [lib/pcd/LegalEntityKindCodeList.js](#libpcdlegalentitykindcodelistjs)   |
| childToiletStatus                          | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| chome                                      | [lib/ex/string.js](#libexstringjs)                                       |
| city                                       | [lib/ex/string.js](#libexstringjs)                                       |
| cityAndCounty                              | [lib/ex/string.js](#libexstringjs)                                       |
| cityAndCountyEn                            | [lib/ex/stringEn.js](#libexstringenjs)                                   |
| cityAndCountyKana                          | [lib/ex/stringKana.js](#libexstringkanajs)                               |
| cityBlock                                  | [lib/ex/string.js](#libexstringjs)                                       |
| closingHours                               | [lib/ex/time.js](#libextimejs)                                           |
| code                                       | [lib/ex/string.js](#libexstringjs)                                       |
| codeType                                   | [lib/ex/string.js](#libexstringjs)                                       |
| contactPointEmailAddress                   | [lib/ex/email.js](#libexemailjs)                                         |
| contactPointName                           | [lib/ex/string.js](#libexstringjs)                                       |
| contactPointPhoneNumber                    | [lib/ex/phone.js](#libexphonejs)                                         |
| contactPointPurpose                        | [lib/ex/string.js](#libexstringjs)                                       |
| contactPointRemark                         | [lib/ex/string.js](#libexstringjs)                                       |
| contactPointSocialMedia                    | [lib/ex/socialMedia.js](#libexsocialmediajs)                             |
| contactPointUrl                            | [lib/ex/uri.js](#libexurijs)                                             |
| contentsUrl                                | [lib/ex/uri.js](#libexurijs)                                             |
| coordinateReferenceSystem                  | [lib/ex/string.js](#libexstringjs)                                       |
| coordinateReferenceSystemCode              | [lib/ex/epsg.js](#libexepsgjs)                                           |
| coorganizer                                | [lib/ex/string.js](#libexstringjs)                                       |
| country                                    | [lib/pcd/CountryCodeList.js](#libpcdcountrycodelistjs)                   |
| countryOfBirth                             | [lib/pcd/CountryCodeList.js](#libpcdcountrycodelistjs)                   |
| county                                     | [lib/ex/string.js](#libexstringjs)                                       |
| dateOfBirth                                | [lib/ex/date.js](#libexdatejs)                                           |
| dateOfCompletion                           | [lib/ex/date.js](#libexdatejs)                                           |
| dateOfDeath                                | [lib/ex/date.js](#libexdatejs)                                           |
| dateOfEnd                                  | [lib/ex/date.js](#libexdatejs)                                           |
| dateOfEstablishment                        | [lib/ex/date.js](#libexdatejs)                                           |
| dateOfStart                                | [lib/ex/date.js](#libexdatejs)                                           |
| dateTimeDescription                        | [lib/ex/string.js](#libexstringjs)                                       |
| deadlineDateOfApplication                  | [lib/ex/date.js](#libexdatejs)                                           |
| description                                | [lib/ex/string.js](#libexstringjs)                                       |
| diaperChangingCornerStatus                 | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| displayedAddressFlag                       | [lib/ex/integer.js](#libexintegerjs)                                     |
| endDateOfPublication                       | [lib/ex/date.js](#libexdatejs)                                           |
| endTime                                    | [lib/ex/time.js](#libextimejs)                                           |
| eventName                                  | [lib/ex/string.js](#libexstringjs)                                       |
| eventNameEn                                | [lib/ex/hankaku.js](#libexhankakujs)                                     |
| eventNameKana                              | [lib/ex/zenkakuKana.js](#libexzenkakukanajs)                             |
| eventType                                  | [lib/ex/string.js](#libexstringjs)                                       |
| eventUrl                                   | [lib/ex/uri.js](#libexurijs)                                             |
| facilityDescription                        | [lib/ex/string.js](#libexstringjs)                                       |
| familyName                                 | [lib/ex/string.js](#libexstringjs)                                       |
| familyNameEn                               | [lib/ex/hankakuAlphaNumeric.js](#libexhankakualphanumericjs)             |
| familyNameKana                             | [lib/ex/zenkakuKana.js](#libexzenkakukanajs)                             |
| familyRegisterFamilyName                   | [lib/ex/string.js](#libexstringjs)                                       |
| familyRegisterGivenName                    | [lib/ex/string.js](#libexstringjs)                                       |
| familyRegisterName                         | [lib/ex/string.js](#libexstringjs)                                       |
| feeDescription                             | [lib/ex/string.js](#libexstringjs)                                       |
| feeStatus                                  | [lib/ex/string.js](#libexstringjs)                                       |
| fee_JPY                                    | [lib/ex/integer.js](#libexintegerjs)                                     |
| floorName                                  | [lib/ex/string.js](#libexstringjs)                                       |
| foodAndDrinkPermissionStatus               | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| foreignLanguageSupport                     | [lib/ex/languageCode.js](#libexlanguagecodejs)                           |
| foreignLanguageSupportRemark               | [lib/ex/string.js](#libexstringjs)                                       |
| formerFamilyName                           | [lib/ex/string.js](#libexstringjs)                                       |
| formerFamilyNameEn                         | [lib/ex/hankakuAlphaNumeric.js](#libexhankakualphanumericjs)             |
| formerFamilyNameKana                       | [lib/ex/zenkakuKana.js](#libexzenkakukanajs)                             |
| fullAddress                                | [lib/ex/string.js](#libexstringjs)                                       |
| fullAddressEn                              | [lib/ex/stringEn.js](#libexstringenjs)                                   |
| fullAddressKana                            | [lib/ex/stringKana.js](#libexstringkanajs)                               |
| fullName                                   | [lib/ex/string.js](#libexstringjs)                                       |
| fullNameEn                                 | [lib/ex/hankaku.js](#libexhankakujs)                                     |
| fullNameKana                               | [lib/ex/spaceSeparatedZenkakuKana.js](#libexspaceseparatedzenkakukanajs) |
| functionalSupportNecessity                 | [lib/ex/string.js](#libexstringjs)                                       |
| functionalSupportType                      | [lib/ex/string.js](#libexstringjs)                                       |
| generalUserAcceptanceStatus                | [lib/ex/string.js](#libexstringjs)                                       |
| givenName                                  | [lib/ex/spaceSeparatedString.js](#libexspaceseparatedstringjs)           |
| givenNameEn                                | [lib/ex/hankakuAlphaNumeric.js](#libexhankakualphanumericjs)             |
| givenNameKana                              | [lib/ex/zenkakuKana.js](#libexzenkakukanajs)                             |
| headOfHousehold                            | [lib/ex/string.js](#libexstringjs)                                       |
| height_CMT                                 | [lib/ex/number.js](#libexnumberjs)                                       |
| height_MTR                                 | [lib/ex/number.js](#libexnumberjs)                                       |
| holdingCondition                           | [lib/ex/string.js](#libexstringjs)                                       |
| holdingPattern                             | [lib/ex/string.js](#libexstringjs)                                       |
| id                                         | [lib/ex/hankaku.js](#libexhankakujs)                                     |
| idType                                     | [lib/ex/string.js](#libexstringjs)                                       |
| incomeStatus                               | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| keyword                                    | [lib/ex/semicolonSeparatedString.js](#libexsemicolonseparatedstringjs)   |
| landArea_MTK                               | [lib/ex/number.js](#libexnumberjs)                                       |
| landForm                                   | [lib/ex/string.js](#libexstringjs)                                       |
| landFormImageUrl                           | [lib/ex/uri.js](#libexurijs)                                             |
| landNumber                                 | [lib/ex/string.js](#libexstringjs)                                       |
| languageCode                               | [lib/ex/languageCode.js](#libexlanguagecodejs)                           |
| largerSectionTown                          | [lib/ex/string.js](#libexstringjs)                                       |
| latitudeOfRepresentativePoint              | [lib/ex/number.js](#libexnumberjs)                                       |
| legalEntityNumber                          | [lib/ex/legalEntityNumber.js](#libexlegalentitynumberjs)                 |
| localGovernmentCode                        | [lib/ex/standardAreaCode.js](#libexstandardareacodejs)                   |
| location                                   | [lib/ex/string.js](#libexstringjs)                                       |
| locationOfReception                        | [lib/ex/string.js](#libexstringjs)                                       |
| locatorName                                | [lib/ex/string.js](#libexstringjs)                                       |
| longitudeOfRepresentativePoint             | [lib/ex/number.js](#libexnumberjs)                                       |
| maritalStatus                              | [lib/pcd/MaritalStatusCodeList.js](#libpcdmaritalstatuscodelistjs)       |
| maxAge                                     | [lib/ex/integer.js](#libexintegerjs)                                     |
| middleName                                 | [lib/ex/string.js](#libexstringjs)                                       |
| middleNameEn                               | [lib/ex/hankakuAlphaNumeric.js](#libexhankakualphanumericjs)             |
| middleNameKana                             | [lib/ex/zenkakuKana.js](#libexzenkakukanajs)                             |
| minAge                                     | [lib/ex/integer.js](#libexintegerjs)                                     |
| minAgeInMonths                             | [lib/ex/string.js](#libexstringjs)                                       |
| mobilitySupportForBrailleBlocksEtcStatus   | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| multipurposeToiletStatus                   | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| name                                       | [lib/ex/string.js](#libexstringjs)                                       |
| nameEn                                     | [lib/ex/stringEn.js](#libexstringenjs)                                   |
| nameKana                                   | [lib/ex/stringKana.js](#libexstringkanajs)                               |
| nationality                                | [lib/pcd/CountryCodeList.js](#libpcdcountrycodelistjs)                   |
| numberOfCompanyEmployee                    | [lib/ex/integer.js](#libexintegerjs)                                     |
| numberOfEmployee                           | [lib/ex/integer.js](#libexintegerjs)                                     |
| numberOfRegularEmployee                    | [lib/ex/integer.js](#libexintegerjs)                                     |
| nurseryStatus                              | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| openingHours                               | [lib/ex/time.js](#libextimejs)                                           |
| organizer                                  | [lib/ex/string.js](#libexstringjs)                                       |
| parentship                                 | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| parkingFeeStatus                           | [lib/ex/string.js](#libexstringjs)                                       |
| parkingInformation                         | [lib/ex/string.js](#libexstringjs)                                       |
| parkingStatus                              | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| paymentType                                | [lib/ex/semicolonSeparatedString.js](#libexsemicolonseparatedstringjs)   |
| performanceDescription                     | [lib/ex/string.js](#libexstringjs)                                       |
| positionOfOrganizationTypeInName           | [lib/ex/string.js](#libexstringjs)                                       |
| positionOfRepresentative                   | [lib/ex/string.js](#libexstringjs)                                       |
| postalCode                                 | [lib/ex/hankaku.js](#libexhankakujs)                                     |
| prefecture                                 | [lib/ex/string.js](#libexstringjs)                                       |
| prefectureEn                               | [lib/ex/stringEn.js](#libexstringenjs)                                   |
| prefectureKana                             | [lib/ex/stringKana.js](#libexstringkanajs)                               |
| priorityParkingStatus                      | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| rampsOrElevatorsOrEscalatorsStatus         | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| relatedGroup                               | [lib/ex/string.js](#libexstringjs)                                       |
| relatedLegalEntityNumber                   | [lib/ex/legalEntityNumber.js](#libexlegalentitynumberjs)                 |
| remark                                     | [lib/ex/string.js](#libexstringjs)                                       |
| residenceNumber                            | [lib/ex/string.js](#libexstringjs)                                       |
| serviceAreaCityCode                        | [lib/ex/standardAreaCode.js](#libexstandardareacodejs)                   |
| serviceAreaPolygon                         | [lib/ex/uri.js](#libexurijs)                                             |
| serviceAreaPrefectureCode                  | [lib/ex/prefectureCode.js](#libexprefecturecodejs)                       |
| serviceAreaRemark                          | [lib/ex/string.js](#libexstringjs)                                       |
| serviceAreaSmallArea                       | [lib/ex/string.js](#libexstringjs)                                       |
| sex                                        | [lib/pcd/SexCodeList.js](#libpcdsexcodelistjs)                           |
| shelterInformation                         | [lib/ex/string.js](#libexstringjs)                                       |
| smallAdministrativeUnit                    | [lib/ex/string.js](#libexstringjs)                                       |
| spouseStatus                               | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| standardFee_JPY                            | [lib/ex/integer.js](#libexintegerjs)                                     |
| startDateOfApplication                     | [lib/ex/date.js](#libexdatejs)                                           |
| startDateOfFiscalYear                      | [lib/ex/monthDate.js](#libexmonthdatejs)                                 |
| startDateOfPublication                     | [lib/ex/date.js](#libexdatejs)                                           |
| startTime                                  | [lib/ex/time.js](#libextimejs)                                           |
| status                                     | [lib/ex/string.js](#libexstringjs)                                       |
| storeysAboveGround                         | [lib/ex/integer.js](#libexintegerjs)                                     |
| storeysBelowGround                         | [lib/ex/integer.js](#libexintegerjs)                                     |
| streetAddress                              | [lib/ex/string.js](#libexstringjs)                                       |
| streetAddressAndCityBlock                  | [lib/ex/string.js](#libexstringjs)                                       |
| streetAddressAndCityBlockEn                | [lib/ex/stringEn.js](#libexstringenjs)                                   |
| streetAddressAndCityBlockKana              | [lib/ex/stringKana.js](#libexstringkanajs)                               |
| streetAddressId                            | [lib/ex/hankaku.js](#libexhankakujs)                                     |
| strollerPermissionStatus                   | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| strollerRentalStatus                       | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| studentStatus                              | [lib/pcd/ApplicabilityCodeList.js](#libpcdapplicabilitycodelistjs)       |
| subEvent                                   | [lib/ex/string.js](#libexstringjs)                                       |
| subTitle                                   | [lib/ex/string.js](#libexstringjs)                                       |
| subtitlesStatus                            | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| tag                                        | [lib/ex/semicolonSeparatedString.js](#libexsemicolonseparatedstringjs)   |
| targetIndustry                             | [lib/ex/string.js](#libexstringjs)                                       |
| targetPersons                              | [lib/ex/string.js](#libexstringjs)                                       |
| targetRemark                               | [lib/ex/string.js](#libexstringjs)                                       |
| timeRequired                               | [lib/ex/time.js](#libextimejs)                                           |
| toiletForOstomatesStatus                   | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| totalFloorArea_MTK                         | [lib/ex/number.js](#libexnumberjs)                                       |
| tradeName                                  | [lib/ex/string.js](#libexstringjs)                                       |
| tradeNameEn                                | [lib/ex/hankaku.js](#libexhankakujs)                                     |
| tradeNameKana                              | [lib/ex/zenkakuKana.js](#libexzenkakukanajs)                             |
| url                                        | [lib/ex/uri.js](#libexurijs)                                             |
| ward                                       | [lib/ex/string.js](#libexstringjs)                                       |
| webSiteUrl                                 | [lib/ex/uri.js](#libexurijs)                                             |
| webToolAndEnvironment                      | [lib/ex/semicolonSeparatedString.js](#libexsemicolonseparatedstringjs)   |
| webinarStatus                              | [lib/ex/string.js](#libexstringjs)                                       |
| webinarUrl                                 | [lib/ex/uri.js](#libexurijs)                                             |
| weight_KGM                                 | [lib/ex/number.js](#libexnumberjs)                                       |
| wheelchairPermissionStatus                 | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| wheelchairRentalStatus                     | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| writingSupportStatus                       | [lib/pcd/StatusCodeList.js](#libpcdstatuscodelistjs)                     |
| yearOfEstablishment                        | [lib/ex/string.js](#libexstringjs)                                       |

# API

- `lib/*.js` は旧 IMI ツールで整備されていた日付・電話番号・全角カナコンポーネントを継承したものです
- `lib/pcd/*.js` は GIF のコードリストに関する実装を収録しています
- `lib/ex/*.js` は新規に導入された validation / normalization のルールです (一部、 `lib/*.js` への参照を含みます)

## lib/date.js

- ISO8601 に従った `yyyy-mm-dd` 形式だけを valid と判定します
- 全角数字・漢数字を半角数字に変換します
- 和暦は西暦に変換します
- 最終的に `yyyymmdd` あるいは `yyyy[非数字*]m[非数字*]d` を `yyyy-mm-dd` として正規化します

## lib/phonenumber.js

- 与えられた文字列が libphonenumber.js の formatNational メソッド (defaultCountry=JP) で正規化した結果と一致する場合に valid と判定します
- 与えられた文字列が + で始まる場合は formatInternational メソッドの結果で判定します
- 漢数字・全角数字を半角数字に、各種カッコを半角カッコに、長音記号類を半角マイナスに、各種カンマを半角カンマに変換した上で、libphonenumber.js の formatNational (defaultCountry=JP) または formatInternational で処理した結果を正規化結果としてリターンします

## lib/zenkakukana.js

- 半角カナ・結合文字が使用されている場合は invalid と判定します
- 正規表現 `^[ァ-ヴー]*$` にマッチする場合に valid と判定されます
- 文字列は Unicode 正規化 (NFKC) し、全角ひらがなは全角カナに変換されます。その結果が valid である場合に正規化結果としてリターンします。

## lib/pcd/ApplicabilityCodeList.js

- `0`, `1`, `9` のいずれかの文字列である場合に valid と判定します
- 数値は文字列化されて正規化されます
- 文字列は Unicode 正規化 (NFKC) して `0-9` 以外の文字を除去した上で、コードリストの値と一致する場合に正規化結果としてリターンされます

## lib/pcd/CountryCodeList.js

- 正規表現 `^[A-Z]{3}$` にマッチし、かつ、コードリストに登録されている文字列である場合に valid と判定します
- 文字列は Unicode 正規化 (NFKC) および大文字化して A-Z 以外の文字列を除去した上で、コードリスト(ISO3166-1-alpha2 および ISO3166-1-alpha3) に一致する場合、ISO3166-1-alpha3 の値が正規化結果としてリターンされます

## lib/pcd/LegalEntityKindCodeList.js

- `101`, `201`, `301`, `302`, `303`, `304`, `305`, `399`, `401`, `499` のいずれかの文字列の場合に valid と判定します
- 数値は文字列化されて正規化されます
- 文字列は Unicode 正規化 (NFKC) して `0-9` 以外の文字を除去した上で、コードリストの値と一致する場合に正規化結果としてリターンされます

## lib/pcd/MaritalStatusCodeList.js

- `0`, `1`, `2`, `3`, `4`, `9` のいずれかの文字列の場合に valid と判定します
- 数値は文字列化されて正規化されます
- 文字列は Unicode 正規化 (NFKC) して `0-9` 以外の文字を除去した上で、コードリストの値と一致する場合に正規化結果としてリターンされます

## lib/pcd/SexCodeList.js

- `0`, `1`, `2`, `9` のいずれかの文字列の場合に valid と判定します
- 数値は文字列化されて正規化されます
- 文字列は Unicode 正規化 (NFKC) して `0-9` 以外の文字を除去した上で、コードリストの値と一致する場合に正規化結果としてリターンされます

## lib/pcd/StatusCodeList.js

- `0`, `1`, `2`, `9` のいずれかの文字列の場合に valid と判定します
- 数値は文字列化されて正規化されます
- 文字列は Unicode 正規化 (NFKC) して `0-9` 以外の文字を除去した上で、コードリストの値と一致する場合に正規化結果としてリターンされます

## lib/ex/date.js

- [lib/date.js](#libdatejs) の別名です

## lib/ex/dateTimeFormula.js

- [441*コアデータパーツ*日付時刻](https://github.com/JDA-DM/GIF/tree/main/440_%E3%82%B3%E3%82%A2%E3%83%87%E3%83%BC%E3%82%BF%E3%83%91%E3%83%BC%E3%83%84/docx) の時間帯コード(数字 2 文字 00-23,99) をカンマ区切りで列挙したものを valid と判定します
- 全角文字は半角文字に変換します
- `HH:MM` 形式は `MM` 部分を丸め込んだうえで時間帯コード `HH` に変換します
- `HH/HH` や `HH:MM/HH:MM` のような範囲形式は `HH,HH,HH..` のような列挙形式に変換します
- 数字・/ (スラッシュ)・: (コロン) 以外の文字列は , 相当の区切り文字として扱います
- 最終的に `02,03...` のように昇順にソートした形で正規化されます

## lib/ex/day.js

- 正規表現 `/^月?火?水?木?金?土?日?$/` にマッチした場合を valid と判定します
- `月曜日`,`月曜`,`Monday`, `Mon` といったワードは `月` に変換されます
- 火～日 についても同様です
- 変換文字列について「月火水木金土日」以外の文字列は除去されます
- 結果文字列を「月火水木金土日」の順番に沿うようにソートした形で正規化されます

## lib/ex/email.js

- [Ajv の email 用正規表現](https://github.com/ajv-validator/ajv-formats/blob/master/src/formats.ts#L64-L65) にマッチした場合を valid と判定します
- Unicode 正規化 (NFKC) によって正規化します

## lib/ex/epsg.js

- 正規表現 `/^EPSG:[0-9]+$/` にマッチした場合を valid と判定します
- 数値および数字文字列は `EPSG:{value}` として正規化されます
- Unicode 正規化 (NFKC) の結果を大文字に変換し、`EPSG:0123456789` 以外の文字を除去したものについて、正規表現にマッチした場合のみ、それを正規化結果としてリターンします

## lib/ex/hankaku.js

- 正規表現 `/^[ -~]*$/` にマッチした場合を valid と判定します
- 数値は文字列に変換することで正規化されます
- Unicode 正規化 (NFKC) の結果が正規表現にマッチする場合のみ、それを正規化結果としてリターンします

## lib/ex/hankakuAlphaNumeric.js

- 正規表現 `/^[a-zA-Z0-9]*$/` にマッチした場合を valid と判定します
- 数値は文字列に変換することで正規化されます
- Unicode 正規化 (NFKC) の結果が正規表現にマッチする場合のみ、それを正規化結果としてリターンします

## lib/ex/hankakuNumeric.js

- 正規表現 `/^[0-9]*$/` にマッチした場合を valid と判定します
- 数値は文字列に変換することで正規化されます
- Unicode 正規化 (NFKC) の結果が正規表現にマッチする場合のみ、それを正規化結果としてリターンします

## lib/ex/integer.js

- [Number.isInteger](https://developer.mozilla.org/ja/docs/Web/JavaScript/Reference/Global_Objects/Number/isInteger) が true の場合に valid と判定します
- 値を文字列化して Unicode 正規化 (NFKC) した結果を `parseInt` で処理した結果が valid だった場合に正規化結果としてリターンします

## lib/ex/languageCode.js

- 正規表現 `^(?:[a-z]{2})(?:;[a-z]{2})*$` にマッチした場合を valid と判定します
- 値を Unicode 正規化 (NFKC)して小文字変換したものについて、`a-z:` 以外の文字を取り除いたものが正規表現にマッチする場合、それを正規化結果としてリターンします

## lib/ex/legalEntityNumber.js

- 正規表現 `^[0-9]{13}$` にマッチし、かつ、[法人番号:チェックデジットの計算](https://www.houjin-bangou.nta.go.jp/documents/checkdigit.pdf) に適合した場合を valid と判定します
- 値を Unicode 正規化 (NFKC)して `0123456789` 以外の文字を取り除いたものが正規表現にマッチする場合、それを正規化結果としてリターンします

## lib/ex/monthDate.js

- 正規表現 `^([0-9]{2})-([0-9]{2})$` にマッチし、かつ、`mm` 部分が `01-12` / `dd` 部分が `01-31` の場合に valid と判定します
- 値を Unicode 正規化 (NFKC) し、mmdd / mdd / m[非数字+]d[非数字] に合致し、かつ、mm-dd が valid である場合にこれを正規化結果としてリターンします
- 例: １２月１日のような文字列は Unicode 正規化(全角 → 半角)および m[非数字+]d[非数字] ルールにより 12-01 として正規化される

## lib/ex/number.js

- 値の [typeof](https://developer.mozilla.org/ja/docs/Web/JavaScript/Reference/Operators/typeof) が `number` であり、かつ `isNaN` が false の場合に valid と判定します
- 文字列は Unicode 正規化 (NFKC) をさらに [parseInt()](https://developer.mozilla.org/ja/docs/Web/JavaScript/Reference/Global_Objects/parseInt) することで正規化されます

## lib/ex/phone.js

- [lib/phonenumber.js](#libphonenumberjs) の別名です

## lib/ex/prefectureCode.js

- 値が `01` ～ `47` のいずれかの文字列である場合を valid と判定します
- 数値は文字列化され、二桁になるように冒頭に `0` が付与されます
- 文字列は Unicode 正規化 (NFKC) し、二桁になるように冒頭に `0` が付与されます
- これらが valid の場合のみ正規化結果がリターンされます

## lib/ex/semicolonSeparatedString.js

- 半角カナ・結合文字が使用されている場合は invalid と判定します
- 半角セミコロン以外の区切り文字(空白・全角セミコロン) が使用されている場合は invalid と判定します
- 連続する半角セミコロン、末尾の半角セミコロンは invalid と判定します
- 上記に該当しない場合は valid と判定されます
- 文字列は Unicode 正規化 (NFKC) し、 空白・全角セミコロンは半角セミコロンに変換します。その上で連続する半角セミコロンを集約、末尾の半角セミコロンを除去した結果を正規化結果としてリターンします。

## lib/ex/socialMedia.js

- 空文字、あるいは正規表現 `^[^:]+:.+$` にマッチする場合に valid と判定します
- 文字列は Unicode 正規化 (NFKC) したものが valid である場合にそれを正規化結果としてリターンします。

## lib/ex/spaceSeparatedString.js

- 半角カナ・結合文字が使用されている場合は invalid と判定します
- 半角スペース以外の空白文字(タブ、改行等) が使用されている場合は invalid と判定します
- 上記に該当しない場合は valid と判定されます
- 文字列は Unicode 正規化 (NFKC) し、 空白文字はすべて半角スペースに変換します。その上で連続する半角スペースを集約、末尾の半角スペースを除去した結果を正規化結果としてリターンします。

## lib/ex/spaceSeparatedZenkakuKana.js

- 半角カナ・結合文字が使用されている場合は invalid と判定します
- 正規表現 `^[ァ-ヴー ]*$` にマッチする場合に valid と判定されます
- 文字列は Unicode 正規化 (NFKC) し、全角ひらがなは全角カナに変換されます。また、空白文字はすべて半角スペースに変換します。その上で連続する半角スペースを集約、末尾の半角スペースを除去した結果を正規化結果としてリターンします。

## lib/ex/standardAreaCode.js

- 正規表現 `^[0-9]{6}$` にマッチし、かつ、[全国地方公共団体コード仕様](https://www.soumu.go.jp/main_content/000137948.pdf) のチェックデジット仕様に適合する場合に valid と判定します
- 数値は文字列化され、6 桁になるように冒頭に 0 が付与されたものが valid な場合、それを正規化結果としてリターンします
- 文字列は Unicode 正規化 (NFKC) し、 `0123456789` 以外の文字列が除去されます
- その結果文字列が 6 桁の場合はそれが valid であれば正規化結果としてリターンします
- その結果文字列が 5 桁の場合は末尾にチェックデジットを付与、それが valid であれば正規化結果としてリターンします

## lib/ex/string.js

- 文字列であれば valid と判定します
- 数値は文字列化して正規化されます
- 文字列は Unicode 正規化 (NFKC) したものを正規化結果としてリターンします

## lib/ex/stringEn.js

- 全角英数字・全角スペースが使われている場合は invalid と判定します
- そうでない場合は valid と判定します
- 数値は文字列化して正規化されます
- 文字列は Unicode 正規化 (NFKC) したものを正規化結果としてリターンします

## lib/ex/stringKana.js

- 半角カナ・結合文字・全角かなが使用されている場合は invalid と判定します
- そうでない場合は valid と判定されます
- Unicode 正規化 (NFKC) し、全角ひらがなを全角カナに変換したものを正規化結果としてリターンします。

## lib/ex/time.js

- 正規表現 `^([0-9]{2}):([0-9]{2})$` にマッチし、かつ、 `hh` 部が 00 ～ 23 / `mm` 部が 00 ～ 59 の場合に valid と判定します
- hmm あるいは hhmm のような数値・文字列、および、 hh[非数字]mm[非数字] のような文字列は hh:mm として正規化し、これが valid な場合は正規化結果としてリターンされます

## lib/ex/uri.js

- `http:` または `https:` で始まる文字列を valid と判定します
- Unicode 正規化 (NFKC) したものを正規化結果としてリターンします

## lib/ex/zenkakuKana.js

- [lib/zenkakukana.js](#libzenkakukanajs) の別名です

# 開発者向け情報

## インストール

以下の手順でインストール～テスト～カバレージ検査が完了します。

```
$ mkdir work
$ cd work
$ tar xvzf ${SOMEWHERE}/dev-gif-component-tools-util-1.0.0.tgz
$ cd dev-gif-component-tools-util-1.0.0
$ npm install
$ npm test
$ npm run coverage
```

## ユニットテスト

以下の手順で `test/test-*.js` のテスト群を実行します。

```
$ npm test

> @dev-gif-component-tools/util@1.0.0 test
> tape test/test-*.js

TAP version 13
# date#normalize
ok 1 "20190101" を "2019-01-01" に正規化できること
ok 2 "20191231" を "2019-12-31" に正規化できること
ok 3 "2019年12月31日" を "2019-12-31" に正規化できること
ok 4 "2019年1月1日" を "2019-01-01" に正規化できること
...

1..966
# tests 966
# pass  966

# ok

$
```

## カバレージ

以下の手順でカバレージを計測します。

```
$ npm run coverage

> @dev-gif-component-tools/util@1.0.0 coverage
> nyc --check-coverage --lines 100 npm test


> @dev-gif-component-tools/util@1.0.0 test
> tape test/test-*.js

TAP version 13
# date#normalize
ok 1 "20190101" を "2019-01-01" に正規化できること
ok 2 "20191231" を "2019-12-31" に正規化できること
ok 3 "2019年12月31日" を "2019-12-31" に正規化できること
...

1..966
# tests 966
# pass  966

# ok

-------------------------------|---------|----------|---------|---------|-------------------
File                           | % Stmts | % Branch | % Funcs | % Lines | Uncovered Line #s
-------------------------------|---------|----------|---------|---------|-------------------
All files                      |     100 |      100 |     100 |     100 |
 lib                           |     100 |      100 |     100 |     100 |
  date.js                      |     100 |      100 |     100 |     100 |
  phonenumber.js               |     100 |      100 |     100 |     100 |
  zenkakukana.js               |     100 |      100 |     100 |     100 |
 lib/ex                        |     100 |      100 |     100 |     100 |
  date.js                      |     100 |      100 |     100 |     100 |
  dateTimeFormula.js           |     100 |      100 |     100 |     100 |
  day.js                       |     100 |      100 |     100 |     100 |
  email.js                     |     100 |      100 |     100 |     100 |
  epsg.js                      |     100 |      100 |     100 |     100 |
  hankaku.js                   |     100 |      100 |     100 |     100 |
  hankakuAlphaNumeric.js       |     100 |      100 |     100 |     100 |
  hankakuNumeric.js            |     100 |      100 |     100 |     100 |
  integer.js                   |     100 |      100 |     100 |     100 |
  languageCode.js              |     100 |      100 |     100 |     100 |
  legalEntityNumber.js         |     100 |      100 |     100 |     100 |
  monthDate.js                 |     100 |      100 |     100 |     100 |
  number.js                    |     100 |      100 |     100 |     100 |
  phone.js                     |     100 |      100 |     100 |     100 |
  prefectureCode.js            |     100 |      100 |     100 |     100 |
  semicolonSeparatedString.js  |     100 |      100 |     100 |     100 |
  socialMedia.js               |     100 |      100 |     100 |     100 |
  spaceSeparatedString.js      |     100 |      100 |     100 |     100 |
  spaceSeparatedZenkakuKana.js |     100 |      100 |     100 |     100 |
  standardAreaCode.js          |     100 |      100 |     100 |     100 |
  string.js                    |     100 |      100 |     100 |     100 |
  stringEn.js                  |     100 |      100 |     100 |     100 |
  stringKana.js                |     100 |      100 |     100 |     100 |
  time.js                      |     100 |      100 |     100 |     100 |
  uri.js                       |     100 |      100 |     100 |     100 |
  zenkakuKana.js               |     100 |      100 |     100 |     100 |
 lib/pcd                       |     100 |      100 |     100 |     100 |
  ApplicabilityCodeList.js     |     100 |      100 |     100 |     100 |
  CountryCodeList.js           |     100 |      100 |     100 |     100 |
  LegalEntityKindCodeList.js   |     100 |      100 |     100 |     100 |
  MaritalStatusCodeList.js     |     100 |      100 |     100 |     100 |
  SexCodeList.js               |     100 |      100 |     100 |     100 |
  StatusCodeList.js            |     100 |      100 |     100 |     100 |
  iso3166.js                   |     100 |      100 |     100 |     100 |
-------------------------------|---------|----------|---------|---------|-------------------
$
```

(以上)
