#!/usr/bin/env node
const fs = require("fs");
const levelup = require("levelup");
const leveldown = require("leveldown");

const http = require("http");

if (process.argv.length < 3 || !process.argv[2].match(/^[1-9][0-9]*$/)) {
  console.error("Usage: node bin/server.js [port number]");
  process.exit(1);
}

const port = parseInt(process.argv[2]);

const server = http.createServer(async (req, res) => {
  if (req.method !== "GET") {
    res.writeHead(405, {
      "Content-Type": "text/plain",
      Allow: "GET",
      "Access-Control-Allow-Origin": "*",
    });
    res.end("405 Method Not Allowed, only GET method is supported");
    return;
  }

  if (req.url === "/") {
    res.writeHead(200, {
      "Content-Type": "text/html; charset=utf-8",
    });
    res.end(fs.readFileSync(`${__dirname}/server.html`, "utf-8"));
    return;
  }

  const re = /^\/([0-9]{13})$/.exec(req.url);
  if (!re) {
    res.writeHead(400, {
      "Content-Type": "text/plain; charset=utf-8",
      "Access-Control-Allow-Origin": "*",
    });
    res.end("400 Bad Request : Only a 13-digit numeric value is allowed.");
    return;
  }

  const key = re[1];
  const db = levelup(leveldown(`${__dirname}/../db`));
  try {
    const str = await db.get(key, {
      asBuffer: false,
    });
    const body = JSON.stringify(JSON.parse(str), null, 2);
    res.writeHead(200, {
      "Content-Type": "application/json; charset=utf-8",
      "Access-Control-Allow-Origin": "*",
    });
    res.end(body);
  } catch (e) {
    console.error(e.message);
    res.writeHead(404, {
      "Content-Type": "text/plain; charset=utf-8",
      "Access-Control-Allow-Origin": "*",
    });
    res.end(`404 Not Found : ${e.message}`);
  } finally {
    try {
      await db.close();
    } catch (e) {}
  }
});
server.listen(port, () => {
  console.log(`@gif-component-tools/legal-entity server is running on port ${port}`);
});
